#!/bin/bash
# Shell script for Packege BIDO
# (C) 2025 AIST All Rights Reserved.

#-----------------
# Default value 
bindir=`pwd`/bin
GNUPLOT=`which gnuplot`
GNUPLOT_VER=`gnuplot -V | awk '{print $2}'`

paramfile=./param.sh
if test "$1" == "-help" -o "$1" == "--help" ;then
    echo "This is a program package for the automatic analysis of microtremor data"
    echo "obtained with a circular array of seismic sensors."
    echo "usage: 1) run.sh [return key]"
    echo "       2) run.sh paramfile [return key]"
    echo "       (paramfile will be created automatically by running this file)"
    exit 0
elif test "$1" != "" ;then
    paramfile_default=$1
    if [ ! -f $paramfile_default ];then
      echo "ERROR: paramfile $paramfile_default not found. STOP."
      echo "(Create a new paramfile, or simply type run.sh [return key].)"
      exit 1
    fi
else
    paramfile_default=./script/default_param.sh
fi
. $paramfile_default

# Check the installation
. ./script/chkinstallation.sh

# subroutine ask()
ask() {
    echo -n "$1"
    read answer
    if [ -n "$answer" ]
    then
        ret=$answer
    else
        ret=$2
    fi
}

#-----------------
# Input values

while [ "$yn" != "y" -a "$yn" != "Y" ]
do
    echo "*****************************************************************"
    echo "                           BIDO2.0"
    echo ""
    echo "This is a program package for the automatic analysis of microtremor data"
    echo "obtained with a circular array of seismic sensors."
    echo "usage: 1) run.sh [return key]"
    echo "       2) run.sh paramfile [return key]"
    echo "       (paramfile will be created automatically by running this file)"
    echo ""
    echo "                           (C) 2025 AIST All Rights Reserved."
    echo "*****************************************************************"
    echo ""
    echo "Before the analysis, you have to have all necessary input data files,"
    echo "each one containing waveform records from a single sensor"
    echo "and having a simple four-column or two-column format as follows:" 
    echo "      time [s]     z(UD)       x(EW)      y(NS)"
    echo "or"
    echo "      time [s]     z(UD)"
    echo "(A space, a tab, or a comma may be used as a separator.)" 
    echo "There is no rule for naming the input data files, except that"
    echo "all of them should be placed in a single directory. The number"
    echo "of the input data files should be equal to the number of"
    echo "seismic sensors in your microtremor array."
    echo "Put dummy (alphanumeric) data to fill the vacant column of z"
    echo "if your data have horizontal (x and y) components only." 
    echo "Are you ready? [y/n] (Type \"n\" to exit the program.)"

    idprepro=0
#    delete_level=1
    
        yn=""
        while [ "$yn" != "y" -a "$yn" != "Y" -a "$yn" != "n" -a "$yn" != "N" ]
        do
          read yn
	  if test "$yn" != "y" -a "$yn" != "Y" -a "$yn" != "n" -a "$yn" != "N" 
	  then
	      echo "Type [y/n]"
          fi
        done
	if test "$yn" == "n" -o "$yn" == "N"  ;then
	    echo Abort.
	    exit 0
        fi
    echo ""
    echo "You can proceed with analysis by entering parameters following instructions."
    echo "Simply press the return key to accept the default value(s)"
    echo "  indicated in brackets."
    echo ""

    if test "$datadir" == "";then
	datadir=`dirname $1`
    fi

    ask "Data directory name [$datadir] >" $datadir
    datadir=$ret
    if [ ! -d $datadir ]
    then
      echo Directory \"$datadir\" not found. STOP.
      exit 1
    fi
    outputdir=$datadir/RESULT
    seismfile=$datadir/seism.d
    segmentfile_z=$outputdir/segment_z.d
    segmentfile_h=$outputdir/segment_h.d
    segmentfile_3c=$outputdir/segment_3c.d

    echo ""
    echo "* Execution parameters"
    ask "Do spectral analysis [e.g., Calculate phase velocities and other properties of surface waves]? (y/n=1/0)[$idcalc] $idcalc_OK" $idcalc
    idcalc=$ret

    if test "$idcalc" == "1" ;then 

#	ask "Preprocess the data? [This includes elimination of the trend (including subtraction of the mean as a special case), application of tapers, bandpass filtering, decimation and correction for differences in instrumental response]  (y/n=1/0)[$idprepro] $idprepro_OK" $idprepro
#    idprepro=$ret

	ask "Automatically select data portions to be used?  (y/n=1/0)[$idmksegment] $idmksegment_OK" $idmksegment 
    idmksegment=$ret

        if test "$idmksegment" == "0";then
	    echo
	    echo "NOTE: The option \"automatic selection of data portions\" is off. You have to manually create a segment file and name it $segmentfile_z, `basename $segmentfile_h` or `basename $segmentfile_3c` according to which data components you are using. See the user manual for details."
	    echo
	fi

	ask "Delete temporary data files? (y/n=1/0)[$delete_level] $delete_level_OK" $delete_level
	delete_level=$ret

    fi

    ask "Plot analysis results? (y/n=1/0)[$idgnplt] $idgnplt_OK" $idgnplt
    idgnplt=$ret

    if test "$idcalc" == "1" ;then 
	echo ""
	echo "* Basic parameters"
	ask "Duration of data segments for the evaluation of spectra [s] [$segment_duration] $segment_duration_OK" $segment_duration
	    segment_duration=$ret

	echo "Number of data segments over which averages are taken."
	ask "  Enter 0 if you wish to use all segments simultaneously [$nseg_segave] $nseg_segave_OK" $nseg_segave
	nseg_segave=$ret

	echo "Band width of the Parzen spectral window."
	ask "  Enter 0 if you don't wish to use a spectral window [Hz] [$smoothband] $smoothband_OK" $smoothband
	smoothband=$ret

	echo ""
	echo "* Data file and array geometry."
	if [ -f $seismfile ]
	then
	    idcreate=0
	    echo "A preexisting seismfile $seismfile"
            echo "(for the data file names and the array geometry) has been detected."
            echo "----------------citation begins here-------------------"
            cat $seismfile | awk '{if($0!=""){print $0}}'
            echo "----------------citation ends here---------------------"
	    problem_in_seismfile=0
	    . ./script/chkseism.sh

            echo "Do you want to use this file as it is? [y/n]"
	    if [ $problem_in_seismfile -eq 1 ];	then
		echo "NOTE: Grammar error(s) detected in seismfile. Type \"y\" to stop the program run.sh (rewrite the seismfile and restart run.sh) or type \"n\" to continue and create a new seismfile interactively."
	    elif [ $problem_in_seismfile -eq 2 ];then
		echo "NOTE: Data file(s) listed in seismfile not found in the directory $datadir. Type \"y\" to stop the program run.sh (rewrite the seismfile or place the data file(s) in $datadir and restart run.sh) or type \"n\" to continue and create a new seismfile interactively."
	    else
                echo "(Type \"n\" to create a new seismfile interactively.)"
	    fi
            yn=""
            while [ "$yn" != "y" -a "$yn" != "Y" -a "$yn" != "n" -a "$yn" != "N" ]
            do
		read yn
		if test "$yn" != "y" -a "$yn" != "Y" -a "$yn" != "n" -a "$yn" != "N" 
   	        then
		    echo "Type [y/n]"
                fi
            done

	    if test "$yn" == "Y" -o "$yn" == "y"  ;then
		if [ $problem_in_seismfile -ge 1 ];then
		    echo exit
		    exit 1
		fi
	    fi
	    if test "$yn" == "n" -o "$yn" == "N"  ;then
		idcreate=1
	    fi
	else
	    idcreate=1
	fi
	if [ $idcreate -eq 1 ];then
            . ./script/make_seismfile.sh
	fi

    fi  # test "$idcalc" 

    . ./script/chkpara.sh

    echo 
    echo "##############################################"
    echo "Data directory name: "$datadir          
    echo ""
    echo "<Execution parameters>"

    echo "Analyses using spectral densities (e.g. calculate properties of surface waves) "$idcalc       $idcalc_OK

    if test "$idcalc" == "1" ;then 
#	echo " * Preprocess the data                                         "$idprepro  $idprepro_OK
	echo " * Automatically select data portions to be used               "$idmksegment  $idmksegment_OK
	echo " * Delete temporary data                                       "$delete_level $delete_level_OK
    fi

    echo "Plot analysis results                                          "$idgnplt      $idgnplt_OK

    if test "$idcalc" == "1" ;then 
	echo ""
	echo "<Basic parameters>"
	echo "Segment duration:                 "$segment_duration "[s]"          $segment_duration_OK 
	echo "Number of segments for averaging: "$nseg_segave                     $nseg_segave_OK
	echo "Parzen window band width:         "$smoothband       "[Hz]"         $smoothband_OK 
    fi
    echo "##############################################"
 
    if [ $idcalc -eq 0 -a $idgnplt -eq 0 ];then
	echo Nothing to be done.
	exit 0
    fi

    if test "$idchkpara_ok" == "0";then
	echo "Incorrect parameter(s) detected."
	echo "Press any key to try again."
        read yn
	yn="n"
    else
	yn=""
	while [ "$yn" != "y" -a "$yn" != "Y" -a "$yn" != "n" -a "$yn" != "N" ]
	do
            echo "Double-check the above parameters."
            echo "Are you sure to start running the program now? [y/n] (Type \"n\" to input the parameters again.)"
            read yn
	done
    fi
done

# Create Input file
echo "#Basic Parameters"                     > $paramfile
#echo "datadir=$datadir"                     >> $paramfile   
echo "segment_duration=$segment_duration"   >> $paramfile   
echo "smoothband=$smoothband"               >> $paramfile   
echo "nseg_segave=$nseg_segave"             >> $paramfile   
echo "#execution"                                  >> $paramfile
#echo "idprepro=$idprepro"                   >> $paramfile
echo "idmksegment=$idmksegment"             >> $paramfile
echo "idcalc=$idcalc"                       >> $paramfile
echo "idgnplt=$idgnplt"                     >> $paramfile
echo "delete_level=$delete_level"           >> $paramfile

# Execution
logfile=$datadir/run.log
if [ -f $logfile ]
then
rm -f $logfile
fi
echo OS `uname`                  > $logfile
echo paramfile:$paramfile       >> $logfile
cat  $paramfile | grep -v '#'   >> $logfile
time . ./script/circle.sh | tee -a $logfile
